"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.KmsKeyringClass = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const helpers_1 = require("./helpers");
const arn_parsing_1 = require("./arn_parsing");
function KmsKeyringClass(BaseKeyring) {
    class KmsKeyring extends BaseKeyring {
        constructor({ clientProvider, generatorKeyId, keyIds = [], grantTokens, discovery, discoveryFilter, }) {
            super();
            /* Precondition: This is an abstract class. (But TypeScript does not have a clean way to model this) */
            (0, material_management_1.needs)(this.constructor !== KmsKeyring, 'new KmsKeyring is not allowed');
            /* Precondition: A noop KmsKeyring is not allowed. */
            (0, material_management_1.needs)(!(!discovery && !generatorKeyId && !keyIds.length), 'Noop keyring is not allowed: Set a keyId or discovery');
            /* Precondition: A keyring can be either a Discovery or have keyIds configured. */
            (0, material_management_1.needs)(!(discovery && (generatorKeyId || keyIds.length)), 'A keyring can be either a Discovery or have keyIds configured.');
            /* Precondition: Discovery filter can only be configured in discovery mode. */
            (0, material_management_1.needs)(discovery || (!discovery && !discoveryFilter), 'Account and partition decrypt filtering are only supported when discovery === true');
            /* Precondition: A Discovery filter *must* be able to match something.
             * I am not going to wait to tell you
             * that no CMK can match an empty account list.
             * e.g. [], [''], '' are not valid.
             */
            (0, material_management_1.needs)(!discoveryFilter ||
                (discoveryFilter.accountIDs &&
                    discoveryFilter.accountIDs.length &&
                    !!discoveryFilter.partition &&
                    discoveryFilter.accountIDs.every((a) => typeof a === 'string' && !!a)), 'A discovery filter must be able to match something.');
            /* Precondition: All KMS key identifiers must be valid. */
            (0, material_management_1.needs)(!generatorKeyId || (0, arn_parsing_1.validAwsKmsIdentifier)(generatorKeyId), 'Malformed arn.');
            (0, material_management_1.needs)(keyIds.every((keyArn) => (0, arn_parsing_1.validAwsKmsIdentifier)(keyArn)), 'Malformed arn.');
            /* Precondition: clientProvider needs to be a callable function. */
            (0, material_management_1.needs)(typeof clientProvider === 'function', 'Missing clientProvider');
            (0, material_management_1.readOnlyProperty)(this, 'clientProvider', clientProvider);
            (0, material_management_1.readOnlyProperty)(this, 'keyIds', Object.freeze(keyIds.slice()));
            (0, material_management_1.readOnlyProperty)(this, 'generatorKeyId', generatorKeyId);
            (0, material_management_1.readOnlyProperty)(this, 'grantTokens', grantTokens);
            (0, material_management_1.readOnlyProperty)(this, 'isDiscovery', !!discovery);
            (0, material_management_1.readOnlyProperty)(this, 'discoveryFilter', discoveryFilter
                ? Object.freeze({
                    ...discoveryFilter,
                    accountIDs: Object.freeze(discoveryFilter.accountIDs),
                })
                : discoveryFilter);
        }
        /* Keyrings *must* preserve the order of EDK's.  The generatorKeyId is the first on this list. */
        async _onEncrypt(material) {
            /* Check for early return (Postcondition): Discovery Keyrings do not encrypt. */
            if (this.isDiscovery)
                return material;
            const keyIds = this.keyIds.slice();
            const { clientProvider, generatorKeyId, grantTokens } = this;
            if (generatorKeyId && !material.hasUnencryptedDataKey) {
                const dataKey = await (0, helpers_1.generateDataKey)(clientProvider, material.suite.keyLengthBytes, generatorKeyId, material.encryptionContext, grantTokens);
                /* Precondition: A generatorKeyId must generate if we do not have an unencrypted data key.
                 * Client supplier is allowed to return undefined if, for example, user wants to exclude particular
                 * regions. But if we are here it means that user configured keyring with a KMS key that was
                 * incompatible with the client supplier in use.
                 */
                if (!dataKey)
                    throw new Error('Generator KMS key did not generate a data key');
                const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY;
                const trace = {
                    keyNamespace: helpers_1.KMS_PROVIDER_ID,
                    keyName: dataKey.KeyId,
                    flags,
                };
                material
                    /* Postcondition: The generated unencryptedDataKey length must match the algorithm specification.
                     * See cryptographic_materials as setUnencryptedDataKey will throw in this case.
                     */
                    .setUnencryptedDataKey(dataKey.Plaintext, trace)
                    .addEncryptedDataKey((0, helpers_1.kmsResponseToEncryptedDataKey)(dataKey), material_management_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX);
            }
            else if (generatorKeyId) {
                keyIds.unshift(generatorKeyId);
            }
            /* Precondition: If a generator does not exist, an unencryptedDataKey *must* already exist.
             * Furthermore *only* CMK's explicitly designated as generators can generate data keys.
             * See cryptographic_materials as getUnencryptedDataKey will throw in this case.
             */
            const unencryptedDataKey = (0, material_management_1.unwrapDataKey)(material.getUnencryptedDataKey());
            const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                material_management_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX;
            for (const kmsKey of keyIds) {
                const kmsEDK = await (0, helpers_1.encrypt)(clientProvider, unencryptedDataKey, kmsKey, material.encryptionContext, grantTokens);
                /* clientProvider may not return a client, in this case there is not an EDK to add */
                if (kmsEDK)
                    material.addEncryptedDataKey((0, helpers_1.kmsResponseToEncryptedDataKey)(kmsEDK), flags);
            }
            return material;
        }
        async _onDecrypt(material, encryptedDataKeys) {
            const keyIds = this.keyIds.slice();
            const { clientProvider, generatorKeyId, grantTokens } = this;
            if (generatorKeyId)
                keyIds.unshift(generatorKeyId);
            /* If there are no key IDs in the list, keyring is in "discovery" mode and will attempt KMS calls with
             * every ARN it comes across in the message. If there are key IDs in the list, it will cross check the
             * ARN it reads with that list before attempting KMS calls. Note that if caller provided key IDs in
             * anything other than a CMK ARN format, the Encryption SDK will not attempt to decrypt those data keys, because
             * the EDK data format always specifies the CMK with the full (non-alias) ARN.
             */
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(keyIds, this));
            const cmkErrors = [];
            for (const edk of decryptableEDKs) {
                let dataKey = false;
                try {
                    dataKey = await (0, helpers_1.decrypt)(clientProvider, edk, material.encryptionContext, grantTokens);
                }
                catch (e) {
                    /* Failures onDecrypt should not short-circuit the process
                     * If the caller does not have access they may have access
                     * through another Keyring.
                     */
                    cmkErrors.push({ errPlus: e });
                }
                /* Check for early return (Postcondition): clientProvider may not return a client. */
                if (!dataKey)
                    continue;
                /* Postcondition: The KeyId from KMS must match the encoded KeyID. */
                (0, material_management_1.needs)(dataKey.KeyId === edk.providerInfo, 'KMS Decryption key does not match the requested key id.');
                const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                const trace = {
                    keyNamespace: helpers_1.KMS_PROVIDER_ID,
                    keyName: dataKey.KeyId,
                    flags,
                };
                /* Postcondition: The decrypted unencryptedDataKey length must match the algorithm specification.
                 * See cryptographic_materials as setUnencryptedDataKey will throw in this case.
                 */
                material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                return material;
            }
            /* Postcondition: A CMK must provide a valid data key or KMS must not have raised any errors.
             * If I have a data key,
             * decrypt errors can be ignored.
             * However, if I was unable to decrypt a data key AND I have errors,
             * these errors should bubble up.
             * Otherwise, the only error customers will see is that
             * the material does not have an unencrypted data key.
             * So I return a concatenated Error message
             */
            (0, material_management_1.needs)(material.hasValidKey() ||
                (!material.hasValidKey() && !cmkErrors.length), cmkErrors.reduce((m, e, i) => `${m} Error #${i + 1} \n ${e.errPlus.stack} \n`, 'Unable to decrypt data key and one or more KMS CMKs had an error. \n '));
            return material;
        }
    }
    (0, material_management_1.immutableClass)(KmsKeyring);
    return KmsKeyring;
}
exports.KmsKeyringClass = KmsKeyringClass;
function filterEDKs(keyIds, { isDiscovery, discoveryFilter }) {
    return function filter({ providerId, providerInfo }) {
        /* Check for early return (Postcondition): Only AWS KMS EDK should be attempted. */
        if (providerId !== helpers_1.KMS_PROVIDER_ID)
            return false;
        /* Discovery keyrings can not have keyIds configured,
         * and non-discovery keyrings must have keyIds configured.
         */
        if (isDiscovery) {
            /* Check for early return (Postcondition): There is no discoveryFilter to further condition discovery. */
            if (!discoveryFilter)
                return true;
            const parsedArn = (0, arn_parsing_1.parseAwsKmsKeyArn)(providerInfo);
            /* Postcondition: Provider info is a well formed AWS KMS ARN. */
            (0, material_management_1.needs)(parsedArn, 'Malformed arn in provider info.');
            /* If the providerInfo is an invalid ARN this will throw.
             * But, this function is also used to extract regions
             * from an CMK to generate a regional client.
             * This means it will NOT throw for something
             * that looks like a bare alias or key id.
             * However, these constructions will not have an account or partition.
             */
            const { AccountId, Partition } = parsedArn;
            /* Postcondition: The account and partition *must* match the discovery filter.
             * Since we are offering a runtime discovery of CMKs
             * it is best to have some form of filter on this.
             * The providerInfo is a CMK ARN and will have the account and partition.
             * By offering these levers customers can easily bound
             * the CMKs to ones they control without knowing the CMKs
             * when the AWS KMS Keyring is instantiated.
             */
            return (discoveryFilter.partition === Partition &&
                discoveryFilter.accountIDs.some((a) => a === AccountId));
        }
        else {
            /* Postcondition: The EDK CMK (providerInfo) *must* match a configured CMK. */
            return keyIds.includes(providerInfo);
        }
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX2tleXJpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMva21zX2tleXJpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7OztBQUl0Qyx5RUFjd0M7QUFDeEMsdUNBTWtCO0FBQ2xCLCtDQUF3RTtBQXlDeEUsU0FBZ0IsZUFBZSxDQUc3QixXQUFnQztJQUNoQyxNQUFNLFVBQVcsU0FBUSxXQUFXO1FBV2xDLFlBQVksRUFDVixjQUFjLEVBQ2QsY0FBYyxFQUNkLE1BQU0sR0FBRyxFQUFFLEVBQ1gsV0FBVyxFQUNYLFNBQVMsRUFDVCxlQUFlLEdBQ1M7WUFDeEIsS0FBSyxFQUFFLENBQUE7WUFDUCx1R0FBdUc7WUFDdkcsSUFBQSwyQkFBSyxFQUFDLElBQUksQ0FBQyxXQUFXLEtBQUssVUFBVSxFQUFFLCtCQUErQixDQUFDLENBQUE7WUFDdkUscURBQXFEO1lBQ3JELElBQUEsMkJBQUssRUFDSCxDQUFDLENBQUMsQ0FBQyxTQUFTLElBQUksQ0FBQyxjQUFjLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQ2xELHVEQUF1RCxDQUN4RCxDQUFBO1lBQ0Qsa0ZBQWtGO1lBQ2xGLElBQUEsMkJBQUssRUFDSCxDQUFDLENBQUMsU0FBUyxJQUFJLENBQUMsY0FBYyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUNqRCxnRUFBZ0UsQ0FDakUsQ0FBQTtZQUNELDhFQUE4RTtZQUM5RSxJQUFBLDJCQUFLLEVBQ0gsU0FBUyxJQUFJLENBQUMsQ0FBQyxTQUFTLElBQUksQ0FBQyxlQUFlLENBQUMsRUFDN0Msb0ZBQW9GLENBQ3JGLENBQUE7WUFDRDs7OztlQUlHO1lBQ0gsSUFBQSwyQkFBSyxFQUNILENBQUMsZUFBZTtnQkFDZCxDQUFDLGVBQWUsQ0FBQyxVQUFVO29CQUN6QixlQUFlLENBQUMsVUFBVSxDQUFDLE1BQU07b0JBQ2pDLENBQUMsQ0FBQyxlQUFlLENBQUMsU0FBUztvQkFDM0IsZUFBZSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQzlCLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxRQUFRLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FDcEMsQ0FBQyxFQUNOLHFEQUFxRCxDQUN0RCxDQUFBO1lBRUQsMERBQTBEO1lBQzFELElBQUEsMkJBQUssRUFDSCxDQUFDLGNBQWMsSUFBSSxJQUFBLG1DQUFxQixFQUFDLGNBQWMsQ0FBQyxFQUN4RCxnQkFBZ0IsQ0FDakIsQ0FBQTtZQUNELElBQUEsMkJBQUssRUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxJQUFBLG1DQUFxQixFQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQ3ZELGdCQUFnQixDQUNqQixDQUFBO1lBQ0QsbUVBQW1FO1lBQ25FLElBQUEsMkJBQUssRUFBQyxPQUFPLGNBQWMsS0FBSyxVQUFVLEVBQUUsd0JBQXdCLENBQUMsQ0FBQTtZQUVyRSxJQUFBLHNDQUFnQixFQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsQ0FBQTtZQUN4RCxJQUFBLHNDQUFnQixFQUFDLElBQUksRUFBRSxRQUFRLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFBO1lBQy9ELElBQUEsc0NBQWdCLEVBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxDQUFBO1lBQ3hELElBQUEsc0NBQWdCLEVBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxXQUFXLENBQUMsQ0FBQTtZQUNsRCxJQUFBLHNDQUFnQixFQUFDLElBQUksRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFBO1lBQ2xELElBQUEsc0NBQWdCLEVBQ2QsSUFBSSxFQUNKLGlCQUFpQixFQUNqQixlQUFlO2dCQUNiLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO29CQUNaLEdBQUcsZUFBZTtvQkFDbEIsVUFBVSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQztpQkFDdEQsQ0FBQztnQkFDSixDQUFDLENBQUMsZUFBZSxDQUNwQixDQUFBO1FBQ0gsQ0FBQztRQUVELGlHQUFpRztRQUNqRyxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQStCO1lBQzlDLGdGQUFnRjtZQUNoRixJQUFJLElBQUksQ0FBQyxXQUFXO2dCQUFFLE9BQU8sUUFBUSxDQUFBO1lBRXJDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLENBQUE7WUFDbEMsTUFBTSxFQUFFLGNBQWMsRUFBRSxjQUFjLEVBQUUsV0FBVyxFQUFFLEdBQUcsSUFBSSxDQUFBO1lBQzVELElBQUksY0FBYyxJQUFJLENBQUMsUUFBUSxDQUFDLHFCQUFxQixFQUFFO2dCQUNyRCxNQUFNLE9BQU8sR0FBRyxNQUFNLElBQUEseUJBQWUsRUFDbkMsY0FBYyxFQUNkLFFBQVEsQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUM3QixjQUFjLEVBQ2QsUUFBUSxDQUFDLGlCQUFpQixFQUMxQixXQUFXLENBQ1osQ0FBQTtnQkFDRDs7OzttQkFJRztnQkFDSCxJQUFJLENBQUMsT0FBTztvQkFDVixNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUE7Z0JBRWxFLE1BQU0sS0FBSyxHQUNULHNDQUFnQixDQUFDLCtCQUErQjtvQkFDaEQsc0NBQWdCLENBQUMsMkJBQTJCO29CQUM1QyxzQ0FBZ0IsQ0FBQywrQkFBK0IsQ0FBQTtnQkFDbEQsTUFBTSxLQUFLLEdBQWlCO29CQUMxQixZQUFZLEVBQUUseUJBQWU7b0JBQzdCLE9BQU8sRUFBRSxPQUFPLENBQUMsS0FBSztvQkFDdEIsS0FBSztpQkFDTixDQUFBO2dCQUVELFFBQVE7b0JBQ047O3VCQUVHO3FCQUNGLHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDO3FCQUMvQyxtQkFBbUIsQ0FDbEIsSUFBQSx1Q0FBNkIsRUFBQyxPQUFPLENBQUMsRUFDdEMsc0NBQWdCLENBQUMsK0JBQStCO29CQUM5QyxzQ0FBZ0IsQ0FBQywyQkFBMkIsQ0FDL0MsQ0FBQTthQUNKO2lCQUFNLElBQUksY0FBYyxFQUFFO2dCQUN6QixNQUFNLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFBO2FBQy9CO1lBRUQ7OztlQUdHO1lBQ0gsTUFBTSxrQkFBa0IsR0FBRyxJQUFBLG1DQUFhLEVBQUMsUUFBUSxDQUFDLHFCQUFxQixFQUFFLENBQUMsQ0FBQTtZQUUxRSxNQUFNLEtBQUssR0FDVCxzQ0FBZ0IsQ0FBQywrQkFBK0I7Z0JBQ2hELHNDQUFnQixDQUFDLDJCQUEyQixDQUFBO1lBQzlDLEtBQUssTUFBTSxNQUFNLElBQUksTUFBTSxFQUFFO2dCQUMzQixNQUFNLE1BQU0sR0FBRyxNQUFNLElBQUEsaUJBQU8sRUFDMUIsY0FBYyxFQUNkLGtCQUFrQixFQUNsQixNQUFNLEVBQ04sUUFBUSxDQUFDLGlCQUFpQixFQUMxQixXQUFXLENBQ1osQ0FBQTtnQkFFRCxxRkFBcUY7Z0JBQ3JGLElBQUksTUFBTTtvQkFDUixRQUFRLENBQUMsbUJBQW1CLENBQzFCLElBQUEsdUNBQTZCLEVBQUMsTUFBTSxDQUFDLEVBQ3JDLEtBQUssQ0FDTixDQUFBO2FBQ0o7WUFFRCxPQUFPLFFBQVEsQ0FBQTtRQUNqQixDQUFDO1FBRUQsS0FBSyxDQUFDLFVBQVUsQ0FDZCxRQUErQixFQUMvQixpQkFBcUM7WUFFckMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsQ0FBQTtZQUNsQyxNQUFNLEVBQUUsY0FBYyxFQUFFLGNBQWMsRUFBRSxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUE7WUFDNUQsSUFBSSxjQUFjO2dCQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUE7WUFFbEQ7Ozs7O2VBS0c7WUFDSCxNQUFNLGVBQWUsR0FBRyxpQkFBaUIsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFBO1lBRTFFLE1BQU0sU0FBUyxHQUFnQixFQUFFLENBQUE7WUFFakMsS0FBSyxNQUFNLEdBQUcsSUFBSSxlQUFlLEVBQUU7Z0JBQ2pDLElBQUksT0FBTyxHQUFvQyxLQUFLLENBQUE7Z0JBQ3BELElBQUk7b0JBQ0YsT0FBTyxHQUFHLE1BQU0sSUFBQSxpQkFBTyxFQUNyQixjQUFjLEVBQ2QsR0FBRyxFQUNILFFBQVEsQ0FBQyxpQkFBaUIsRUFDMUIsV0FBVyxDQUNaLENBQUE7aUJBQ0Y7Z0JBQUMsT0FBTyxDQUFDLEVBQUU7b0JBQ1Y7Ozt1QkFHRztvQkFDSCxTQUFTLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUE7aUJBQy9CO2dCQUVELHFGQUFxRjtnQkFDckYsSUFBSSxDQUFDLE9BQU87b0JBQUUsU0FBUTtnQkFFdEIscUVBQXFFO2dCQUNyRSxJQUFBLDJCQUFLLEVBQ0gsT0FBTyxDQUFDLEtBQUssS0FBSyxHQUFHLENBQUMsWUFBWSxFQUNsQyx5REFBeUQsQ0FDMUQsQ0FBQTtnQkFFRCxNQUFNLEtBQUssR0FDVCxzQ0FBZ0IsQ0FBQywrQkFBK0I7b0JBQ2hELHNDQUFnQixDQUFDLDZCQUE2QixDQUFBO2dCQUNoRCxNQUFNLEtBQUssR0FBaUI7b0JBQzFCLFlBQVksRUFBRSx5QkFBZTtvQkFDN0IsT0FBTyxFQUFFLE9BQU8sQ0FBQyxLQUFLO29CQUN0QixLQUFLO2lCQUNOLENBQUE7Z0JBRUQ7O21CQUVHO2dCQUNILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFBO2dCQUN4RCxPQUFPLFFBQVEsQ0FBQTthQUNoQjtZQUVEOzs7Ozs7OztlQVFHO1lBQ0gsSUFBQSwyQkFBSyxFQUNILFFBQVEsQ0FBQyxXQUFXLEVBQUU7Z0JBQ3BCLENBQUMsQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLEVBQ2hELFNBQVMsQ0FBQyxNQUFNLENBQ2QsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssS0FBSyxFQUM1RCx1RUFBdUUsQ0FDeEUsQ0FDRixDQUFBO1lBRUQsT0FBTyxRQUFRLENBQUE7UUFDakIsQ0FBQztLQUNGO0lBQ0QsSUFBQSxvQ0FBYyxFQUFDLFVBQVUsQ0FBQyxDQUFBO0lBQzFCLE9BQU8sVUFBVSxDQUFBO0FBQ25CLENBQUM7QUFyUEQsMENBcVBDO0FBRUQsU0FBUyxVQUFVLENBR2pCLE1BQWdCLEVBQUUsRUFBRSxXQUFXLEVBQUUsZUFBZSxFQUF5QjtJQUN6RSxPQUFPLFNBQVMsTUFBTSxDQUFDLEVBQUUsVUFBVSxFQUFFLFlBQVksRUFBb0I7UUFDbkUsbUZBQW1GO1FBQ25GLElBQUksVUFBVSxLQUFLLHlCQUFlO1lBQUUsT0FBTyxLQUFLLENBQUE7UUFDaEQ7O1dBRUc7UUFDSCxJQUFJLFdBQVcsRUFBRTtZQUNmLHlHQUF5RztZQUN6RyxJQUFJLENBQUMsZUFBZTtnQkFBRSxPQUFPLElBQUksQ0FBQTtZQUVqQyxNQUFNLFNBQVMsR0FBRyxJQUFBLCtCQUFpQixFQUFDLFlBQVksQ0FBQyxDQUFBO1lBQ2pELGdFQUFnRTtZQUNoRSxJQUFBLDJCQUFLLEVBQUMsU0FBUyxFQUFFLGlDQUFpQyxDQUFDLENBQUE7WUFDbkQ7Ozs7OztlQU1HO1lBQ0gsTUFBTSxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsR0FBRyxTQUFTLENBQUE7WUFDMUM7Ozs7Ozs7ZUFPRztZQUNILE9BQU8sQ0FDTCxlQUFlLENBQUMsU0FBUyxLQUFLLFNBQVM7Z0JBQ3ZDLGVBQWUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEtBQUssU0FBUyxDQUFDLENBQ3hELENBQUE7U0FDRjthQUFNO1lBQ0wsOEVBQThFO1lBQzlFLE9BQU8sTUFBTSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQTtTQUNyQztJQUNILENBQUMsQ0FBQTtBQUNILENBQUMifQ==