/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef INCLUDED_ORCUS_JSON_PARSER_THREAD_HPP
#define INCLUDED_ORCUS_JSON_PARSER_THREAD_HPP

#include "env.hpp"
#include "types.hpp"

#include <memory>
#include <vector>
#include <ostream>
#include <variant>

namespace orcus {

class string_pool;

namespace json {

struct ORCUS_PSR_DLLPUBLIC parser_stats
{
    size_t token_buffer_size_threshold;
};

enum class parse_token_t
{
    unknown,
    begin_parse,
    end_parse,
    begin_array,
    end_array,
    begin_object,
    object_key,
    end_object,
    boolean_true,
    boolean_false,
    null,
    string,
    number,
    parse_error,
};

struct ORCUS_PSR_DLLPUBLIC parse_token
{
    using value_type = std::variant<std::string_view, parse_error_value_t, double>;

    parse_token_t type;
    value_type value;

    parse_token();
    parse_token(parse_token_t _type);
    parse_token(parse_token_t _type, std::string_view s);
    parse_token(std::string_view s, std::ptrdiff_t offset);
    parse_token(double value);

    parse_token(const parse_token& other);

    parse_token& operator= (parse_token) = delete;

    bool operator== (const parse_token& other) const;
    bool operator!= (const parse_token& other) const;
};

typedef std::vector<parse_token> parse_tokens_t;

ORCUS_PSR_DLLPUBLIC std::ostream& operator<< (std::ostream& os, const parse_tokens_t& tokens);

class ORCUS_PSR_DLLPUBLIC parser_thread
{
    struct impl;
    std::unique_ptr<impl> mp_impl;

public:
    parser_thread(const char* p, size_t n, size_t min_token_size);
    parser_thread(const char* p, size_t n, size_t min_token_size, size_t max_token_size);
    ~parser_thread();

    void start();

    /**
     * Wait until new set of tokens becomes available.
     *
     * @param tokens new set of tokens.
     *
     * @return true if the parsing is still in progress (therefore more tokens
     *         to come), false if it's done i.e. this is the last token set.
     */
    bool next_tokens(parse_tokens_t& tokens);

    parser_stats get_stats() const;

    void swap_string_pool(string_pool& pool);
};

}}

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
