"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSavedSearchReport = createSavedSearchReport;
var _dataReportHelpers = require("./dataReportHelpers");
var _helpers = require("./helpers");
var _constants = require("./constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Specify how long scroll context should be maintained for scrolled search
 */
const scrollTimeout = '1m';
async function createSavedSearchReport(report, client, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask = true, logger, timezone) {
  const params = report.report_definition.report_params;
  const reportFormat = params.core_params.report_format;
  const reportName = params.report_name;
  await populateMetaData(client, report, isScheduledTask, logger);
  const data = await generateReportData(client, params.core_params, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask, logger, timezone);
  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = (0, _helpers.getFileName)(reportName, curTime) + '.' + reportFormat;
  return {
    timeCreated,
    dataUrl: data,
    fileName
  };
}

/**
 * Populate parameters and saved search info related to meta data object.
 * @param client  OpenSearch client
 * @param report  Report input
 */
async function populateMetaData(client, report, isScheduledTask, logger) {
  _dataReportHelpers.metaData.saved_search_id = report.report_definition.report_params.core_params.saved_search_id;
  _dataReportHelpers.metaData.report_format = report.report_definition.report_params.core_params.report_format;
  _dataReportHelpers.metaData.start = report.time_from;
  _dataReportHelpers.metaData.end = report.time_to;

  // Get saved search info
  let resIndexPattern = {};
  const ssParams = {
    index: '.kibana',
    id: 'search:' + _dataReportHelpers.metaData.saved_search_id
  };
  const ssInfos = await (0, _helpers.callCluster)(client, 'get', ssParams, isScheduledTask);
  _dataReportHelpers.metaData.sorting = ssInfos._source.search.sort;
  _dataReportHelpers.metaData.type = ssInfos._source.type;
  _dataReportHelpers.metaData.searchSourceJSON = ssInfos._source.search.kibanaSavedObjectMeta.searchSourceJSON;

  // Get the list of selected columns in the saved search.Otherwise select all the fields under the _source
  await (0, _dataReportHelpers.getSelectedFields)(ssInfos._source.search.columns);

  // Get index name
  for (const item of ssInfos._source.references) {
    if (item.name === JSON.parse(_dataReportHelpers.metaData.searchSourceJSON).indexRefName) {
      // Get index-pattern information
      const indexPattern = await (0, _helpers.callCluster)(client, 'get', {
        index: '.kibana',
        id: 'index-pattern:' + item.id
      }, isScheduledTask);
      resIndexPattern = indexPattern._source['index-pattern'];
      _dataReportHelpers.metaData.paternName = resIndexPattern.title;
      _dataReportHelpers.metaData.timeFieldName = resIndexPattern.timeFieldName, _dataReportHelpers.metaData.fields = resIndexPattern.fields; // Get all fields
      // Getting fields of type Date
      const dateFields = [];
      for (const item of JSON.parse(_dataReportHelpers.metaData.fields)) {
        if (item.type === 'date') {
          dateFields.push(item.name);
        }
      }
      _dataReportHelpers.metaData.dateFields = dateFields;
    }
  }
}

/**
 * Generate CSV and XLSX data by query and convert OpenSearch data set.
 * @param client  OpenSearch client
 * @param limit   limit size of result data set
 */
async function generateReportData(client, params, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask, logger, timezone) {
  let opensearchData = {};
  const arrayHits = [];
  const report = {
    _source: _dataReportHelpers.metaData
  };
  const indexPattern = report._source.paternName;
  const maxResultSize = await getMaxResultSize();
  const opensearchCount = await getOpenSearchDataSize();
  const total = Math.min(opensearchCount.count, params.limit);
  if (total === 0) {
    return '';
  }
  const reqBody = (0, _dataReportHelpers.buildRequestBody)(report, allowLeadingWildcards, 0);
  logger.info(`[Reporting ${params.report_format} module] DSL request body: ${JSON.stringify(reqBody)}`);
  if (total > maxResultSize) {
    await getOpenSearchDataByScroll();
  } else {
    await getOpenSearchDataBySearch();
  }
  if (params.report_format == _constants.FORMAT.xlsx) {
    return convertOpenSearchDataToExcel();
  }
  return convertOpenSearchDataToCsv();

  // Fetch OpenSearch query max size windows to decide search or scroll
  async function getMaxResultSize() {
    const settings = await (0, _helpers.callCluster)(client, 'indices.getSettings', {
      index: indexPattern,
      includeDefaults: true
    }, isScheduledTask);
    let maxResultSize = Number.MAX_SAFE_INTEGER;
    for (let indexName in settings) {
      // The location of max result window differs if default overridden.
      maxResultSize = Math.min(maxResultSize, settings[indexName].settings.index.max_result_window || settings[indexName].defaults.index.max_result_window);
    }
    return maxResultSize;
  }

  // Build the OpenSearch Count query to count the size of result
  async function getOpenSearchDataSize() {
    const countReq = (0, _dataReportHelpers.buildRequestBody)(report, allowLeadingWildcards, 1);
    return await (0, _helpers.callCluster)(client, 'count', {
      index: indexPattern,
      body: countReq
    }, isScheduledTask);
  }
  async function getOpenSearchDataByScroll() {
    const searchParams = {
      index: report._source.paternName,
      scroll: scrollTimeout,
      body: reqBody,
      size: maxResultSize
    };
    // Open scroll context by fetching first batch
    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits);

    // Start scrolling till the end
    const nbScroll = Math.floor(total / maxResultSize);
    for (let i = 0; i < nbScroll; i++) {
      const resScroll = await (0, _helpers.callCluster)(client, 'scroll', {
        scrollId: opensearchData._scroll_id,
        scroll: scrollTimeout
      }, isScheduledTask);
      if (Object.keys(resScroll.hits.hits).length > 0) {
        arrayHits.push(resScroll.hits);
      }
    }

    // Clear scroll context
    await (0, _helpers.callCluster)(client, 'clearScroll', {
      scrollId: opensearchData._scroll_id
    }, isScheduledTask);
  }
  async function getOpenSearchDataBySearch() {
    const searchParams = {
      index: report._source.paternName,
      body: reqBody,
      size: total
    };
    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits);
  }

  // Parse OpenSearch data and convert to CSV
  async function convertOpenSearchDataToCsv() {
    const dataset = [];
    dataset.push((0, _dataReportHelpers.getOpenSearchData)(arrayHits, report, params, dateFormat, timezone));
    return await (0, _dataReportHelpers.convertToCSV)(dataset, csvSeparator);
  }
  async function convertOpenSearchDataToExcel() {
    const dataset = [];
    dataset.push((0, _dataReportHelpers.getOpenSearchData)(arrayHits, report, params, dateFormat, timezone));
    return await (0, _dataReportHelpers.convertToExcel)(dataset);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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